// Simple i18n with FR/EN dictionaries and a t(key) helper
(() => {
  const translations = {
    fr: {
      french: "Français",
      english: "Anglais",
      login_title: "Connexion",
      email: "Email",
      password: "Mot de passe",
      remember_me: "Se souvenir de moi",
      login: "Se connecter",
      forgot_password: "Mot de passe oublié ?",
      no_account: "Pas encore de compte ?",
      sign_up: "Inscris-toi",
      register_title: "Inscription",
      name: "Nom",
      repeat_password: "Répéter le mot de passe",
      phone: "Téléphone",
      accept_terms: "J'accepte les conditions générales",
      create_account: "Créer mon compte",
      already_registered: "Déjà inscrit ?",
      sign_in: "Connecte-toi",
      logout_title: "Se déconnecter",

      // Main UI
      ai_settings_title: "REGLAGE IA",
      ai_settings_sub: "Personnalise ton double IA",
      style_label: "Style",
      style_placeholder: "Décris ton style...",
      tone_label: "Ton",
      tone_placeholder: "Définis ton ton préféré...",
      forbidden_label: "Interdits",
      forbidden_chip_merci: "Merci pour ce partage",
      forbidden_chip_bravo: "Bravo",
      forbidden_placeholder: "Saisis tes interdits séparés par des virgules...",
      length_label: "Longueur",
      length_short: "Courte",
      length_medium: "Moyenne",
      length_long: "Longue",
      save: "Enregistrer",
      hint_text: "Astuce: si plusieurs options sont activées, Ocalis gère automatiquement les interactions.",
      launch_ocalis: "Lancer Ocalis",
      stop_ocalis: "Stopper Ocalis",

      auto_adjustments_title: "Ajustements automatiques",
      auto_actions_title: "Actions automatiques",
      linebreak_prob_label: "Probabilité de saut de ligne",
      typo_prob_label: "Probabilité de fautes",
      save_adjustments: "Enregistrer",
      auto_like: "Auto Like",
      auto_comment: "Auto Comment",
      auto_reply: "Auto Reply",
      reply_my_posts_only: "Mes postes uniquement",
      comments_per_action: "Commentaires par action",

      // Subscription
      current_plan_label: "Plan actuel :",
      free_plan: "gratuit",
      refresh_plan_title: "Rafraîchir le plan",
      change_plan: "Changer de plan",
      manage_subscription: "Gérer mon abonnement",
      choose_plan: "Choisissez votre plan :",
      billing_monthly: "Mensuel",
      billing_yearly: "Annuel",
      popular_badge: "Populaire",
      plan_pro: "Pro",
      plan_elite: "Elite",
      plan_gold: "Gold",
      plan_desc_pro: "10 commentaires/jour + bonus Likes",
      plan_desc_elite: "50 commentaires/jour + bonus Likes",
      plan_desc_gold: "100 commentaires/jour + bonus Likes",
      cancel: "Annuler",
      per_month: "/mois",
      per_year: "/an",
      expire_on: "Expire le :",

      // Notifications and prompts
      register_success: "Inscription réussie.",
      register_network_error: "Impossible de s'inscrire (erreur réseau)",
      login_success: "Connexion réussie.",
      login_invalid_credentials: "Mot de passe ou email incorrect.",
      invalid_server_response: "Réponse invalide du serveur.",
      login_network_error: "Impossible de se connecter (erreur réseau).",
      email_and_password_required: "Email et mot de passe requis.",
      all_fields_required: "Tous les champs sont requis.",
      invalid_email: "Email invalide.",
      weak_password_message: "Mot de passe faible. Minimum 8 caractères, avec majuscule, minuscule, chiffre et caractère spécial.",
      passwords_mismatch: "Les mots de passe ne correspondent pas.",
      invalid_phone: "Numéro de téléphone invalide.",
      accept_terms_required: "Veuillez accepter les conditions générales.",
      register_user_exists: "Un compte avec cet email existe déjà.",
      register_missing_fields: "Veuillez remplir tous les champs.",
      register_generic_error: "Erreur lors de l'inscription. Veuillez réessayer.",
      please_relogin: "Veuillez vous reconnecter.",
      please_login: "Veuillez vous connecter.",
      confirm_launch: "Êtes-vous sûr de lancer Ocalis ?",
      validate_profile_reply_required: "Veuillez valider le profil IA avant de lancer les réponses.",
      validate_profile_comment_required: "Veuillez valider le profil IA avant de lancer les commentaires.",
      enable_at_least_one_action: "Veuillez activer au moins une action.",
      adjustments_saved: "Ajustements sauvegardés.",
      ai_settings_saved: "Réglage IA enregistré.",
      no_checkout_url_received: "Aucune URL de paiement reçue.",
      upgrade_error: "Erreur lors de la mise à niveau.",
      portal_unavailable: "Portail indisponible.",
      portal_access_error: "Erreur lors de l'accès au portail.",
      quota_unlimited_free_7d: "Quota : illimité gratuitement pendant 7 jours",
      quota_unlimited: "Quota : illimité",
      quota_remaining_today: "Quota restant aujourd'hui : {n} commentaire(s)",
      quota_depleted_alert: "Votre quota journalier est maintenant épuisé (0 commentaire à envoyer). Vous pourrez revenir demain.",
      linkedin_missing_title: "Page LinkedIn introuvable",
      linkedin_missing_desc: "Ouvrir LinkedIn pour utiliser OCALIS.",
      open_here: "Ouvrir ici",

      // Content.js
      generate_with_ocalis: "Générer avec Ocalis",

      // Onboarding
      ob_create_your_ai_clone: "Crée ton double IA",
      ob_subtitle_analyze_profile: "On analyse ton profil LinkedIn pour générer un style psychologique + littéraire unique.",
      ob_btn_analyze: "Analyser mon profil",
      ob_hint_open_profile: "Astuce: ouvre ton profil LinkedIn avant de lancer.",
      ob_loading_title: "Analyse en cours...",
      ob_status_opening_profile: "Ouverture du profil...",
      ob_status_cleaning_texts: "Nettoyage des textes...",
      ob_status_ai_analysis: "Analyse intelligente avec une IA...",
      ob_status_finalizing: "Finalisation...",
      ob_error_cannot_analyze: "Impossible d’analyser.",
      ob_hint_open_profile_again: "Ouvre ton profil LinkedIn puis relance.",
      ob_result_title: "Résultat",
      ob_label_psychology: "Psychologie",
      ob_label_literary: "Littéraire",
      ob_label_forbidden: "Interdits",
      ob_continue: "Continuer",
      ob_validate_title: "Teste ton double IA. Tu veux le garder tel quel ou l’ajuster ?",
      ob_adjust_btn: "Ajuster",
      ob_validate_btn: "Valider",
      ob_edit_subtitle: "Ajuste rapidement ton style d’écriture.",
      ob_label_style: "Style (lexique, structures)",
      ob_placeholder_style: "Ex.: ciné, scène, punchline; phrases courtes, narration 2–3 phrases",
      ob_help_style: "Saisis des mots-clés (séparés par des virgules) et/ou des structures.",
      ob_label_tone: "Ton",
      ob_placeholder_tone: "Ex.: ironique mais bienveillant; narratif",
      ob_help_tone: "Décris le ton général à privilégier.",
      ob_placeholder_forbidden: "mots/expressions à bannir, séparés par des virgules",
      ob_save_btn: "Enregistrer",
      ob_ready_title: "Ton double IA est prêt.",
      ob_ready_subtitle: "À partir de maintenant, chaque commentaire respectera ton style.",
      ob_activate_btn: "Activer Ocalis",
    },
    en: {
      french: "French",
      english: "English",
      login_title: "Sign in",
      email: "Email",
      password: "Password",
      remember_me: "Remember me",
      login: "Log in",
      forgot_password: "Forgot password?",
      no_account: "No account yet?",
      sign_up: "Sign up",
      register_title: "Sign up",
      name: "Name",
      repeat_password: "Repeat password",
      phone: "Phone",
      accept_terms: "I accept the terms",
      create_account: "Create account",
      already_registered: "Already registered?",
      sign_in: "Sign in",
      logout_title: "Log out",

      // Main UI
      ai_settings_title: "AI SETTINGS",
      ai_settings_sub: "Customize your AI twin",
      style_label: "Style",
      style_placeholder: "Describe your style...",
      tone_label: "Tone",
      tone_placeholder: "Set your preferred tone...",
      forbidden_label: "Forbidden",
      forbidden_chip_merci: "Thanks for sharing",
      forbidden_chip_bravo: "Bravo",
      forbidden_placeholder: "Enter forbidden phrases separated by commas...",
      length_label: "Length",
      length_short: "Short",
      length_medium: "Medium",
      length_long: "Long",
      save: "Save",
      hint_text: "Tip: if multiple options are enabled, Ocalis orchestrates interactions automatically.",
      launch_ocalis: "Launch Ocalis",
      stop_ocalis: "Stop Ocalis",

      auto_adjustments_title: "Auto adjustments",
      auto_actions_title: "Automatic actions",
      linebreak_prob_label: "Line break probability",
      typo_prob_label: "Typos probability",
      save_adjustments: "Save",
      auto_like: "Auto Like",
      auto_comment: "Auto Comment",
      auto_reply: "Auto Reply",
      reply_my_posts_only: "My posts only",
      comments_per_action: "Comments per action",

      // Subscription
      current_plan_label: "Current plan:",
      free_plan: "free",
      refresh_plan_title: "Refresh plan",
      change_plan: "Change plan",
      manage_subscription: "Manage subscription",
      choose_plan: "Choose your plan:",
      billing_monthly: "Monthly",
      billing_yearly: "Yearly",
      popular_badge: "Popular",
      plan_pro: "Pro",
      plan_elite: "Elite",
      plan_gold: "Gold",
      plan_desc_pro: "10 comments/day + Likes bonus",
      plan_desc_elite: "50 comments/day + Likes bonus",
      plan_desc_gold: "100 comments/day + Likes bonus",
      cancel: "Cancel",
      per_month: "/month",
      per_year: "/year",
      expire_on: "Expires on:",

      // Notifications and prompts
      register_success: "Registration successful.",
      register_network_error: "Unable to sign up (network error)",
      login_success: "Login successful.",
      login_invalid_credentials: "Incorrect email or password.",
      invalid_server_response: "Invalid server response.",
      login_network_error: "Unable to log in (network error).",
      email_and_password_required: "Email and password required.",
      all_fields_required: "All fields are required.",
      invalid_email: "Invalid email.",
      weak_password_message: "Weak password. Minimum 8 chars with upper, lower, digit and special.",
      passwords_mismatch: "Passwords do not match.",
      invalid_phone: "Invalid phone number.",
      accept_terms_required: "Please accept the terms.",
      register_user_exists: "An account with this email already exists.",
      register_missing_fields: "Please fill in all fields.",
      register_generic_error: "Registration error. Please try again.",
      please_relogin: "Please log in again.",
      please_login: "Please log in.",
      confirm_launch: "Are you sure you want to launch Ocalis?",
      validate_profile_reply_required: "Please validate the AI profile before launching replies.",
      validate_profile_comment_required: "Please validate the AI profile before launching comments.",
      enable_at_least_one_action: "Please enable at least one action.",
      adjustments_saved: "Adjustments saved.",
      ai_settings_saved: "AI settings saved.",
      no_checkout_url_received: "No checkout URL received.",
      upgrade_error: "Upgrade error.",
      portal_unavailable: "Portal unavailable.",
      portal_access_error: "Error accessing portal.",
      quota_unlimited_free_7d: "Quota: unlimited free for 7 days",
      quota_unlimited: "Quota: unlimited",
      quota_remaining_today: "Remaining quota today: {n} comment(s)",
      quota_depleted_alert: "Your daily quota is now exhausted (0 comments to send). You can try again tomorrow.",
      linkedin_missing_title: "LinkedIn page not found",
      linkedin_missing_desc: "Open LinkedIn to use OCALIS.",
      open_here: "Open here",

      // Content.js
      generate_with_ocalis: "Generate with Ocalis",

      // Onboarding
      ob_create_your_ai_clone: "Create your AI twin",
      ob_subtitle_analyze_profile: "We analyze your LinkedIn profile to generate a unique psychological + literary style.",
      ob_btn_analyze: "Analyze my profile",
      ob_hint_open_profile: "Tip: open your LinkedIn profile before starting.",
      ob_loading_title: "Analyzing...",
      ob_status_opening_profile: "Opening profile...",
      ob_status_cleaning_texts: "Cleaning texts...",
      ob_status_ai_analysis: "Smart AI analysis...",
      ob_status_finalizing: "Finalizing...",
      ob_error_cannot_analyze: "Unable to analyze.",
      ob_hint_open_profile_again: "Open your LinkedIn profile then retry.",
      ob_result_title: "Result",
      ob_label_psychology: "Psychology",
      ob_label_literary: "Literary",
      ob_label_forbidden: "Forbidden",
      ob_continue: "Continue",
      ob_validate_title: "Test your AI twin. Keep it as is or adjust it?",
      ob_adjust_btn: "Adjust",
      ob_validate_btn: "Validate",
      ob_edit_subtitle: "Quickly adjust your writing style.",
      ob_label_style: "Style (lexicon, structures)",
      ob_placeholder_style: "Ex.: cinema, scene, punchline; short sentences, 2–3 sentence narration",
      ob_help_style: "Enter keywords (comma-separated) and/or structures.",
      ob_label_tone: "Tone",
      ob_placeholder_tone: "Ex.: ironic but kind; narrative",
      ob_help_tone: "Describe the general tone to favor.",
      ob_placeholder_forbidden: "words/expressions to ban, separated by commas",
      ob_save_btn: "Save",
      ob_ready_title: "Your AI twin is ready.",
      ob_ready_subtitle: "From now on, each comment will follow your style.",
      ob_activate_btn: "Activate Ocalis",
    }
  };

  let currentLang = (localStorage.getItem('lang') || 'fr').toLowerCase();
  if (!translations[currentLang]) currentLang = 'fr';

  // Try to sync with chrome.storage.local on load (non-blocking)
  try {
    if (typeof chrome !== 'undefined' && chrome.storage && chrome.storage.local) {
      chrome.storage.local.get(['lang'], (st) => {
        const saved = (st && st.lang) ? String(st.lang).toLowerCase() : null;
        if (saved && translations[saved] && saved !== currentLang) {
          currentLang = saved;
          // Only apply if our UI container exists
          const root = document.querySelector('[data-ocalis-ui]');
          if (root) applyTranslations(root);
        }
      });
    }
  } catch {}

  function t(key, vars) {
    const pack = translations[currentLang] || translations.fr;
    let val = (pack && pack[key]) || key;
    if (vars && typeof val === 'string') {
      Object.entries(vars).forEach(([k, v]) => {
        val = val.replace(new RegExp('\\{' + k + '\\}', 'g'), String(v));
      });
    }
    return val;
  }

  function applyTranslations(rootNode) {
    const root = rootNode || document.querySelector('[data-ocalis-ui]');
    if (!root) return; // do not touch host pages without our UI
    // Update lang attribute on our UI root only
    try { root.setAttribute('lang', currentLang); } catch {}

    // Elements with data-i18n => textContent
    root.querySelectorAll('[data-i18n]').forEach(el => {
      const k = el.getAttribute('data-i18n');
      el.textContent = t(k);
    });

    // Placeholders
    root.querySelectorAll('[data-i18n-placeholder]').forEach(el => {
      const k = el.getAttribute('data-i18n-placeholder');
      el.setAttribute('placeholder', t(k));
    });

    // Title attributes
    root.querySelectorAll('[data-i18n-title]').forEach(el => {
      const k = el.getAttribute('data-i18n-title');
      el.setAttribute('title', t(k));
    });

    // Update language switch UI
    const flag = document.getElementById('current-lang-flag');
    const label = document.getElementById('current-lang-label');
    if (flag) {
      flag.src = currentLang === 'fr'
        ? 'https://hatscripts.github.io/circle-flags/flags/fr.svg'
        : 'https://hatscripts.github.io/circle-flags/flags/gb.svg';
    }
    if (label) label.textContent = currentLang.toUpperCase();

    // Extra: patch remaining popup labels/placeholders if not tagged
    try {
      // Manage subscription button
      const manageSpan = root.querySelector('#manage-sub-btn span');
      if (manageSpan) manageSpan.textContent = t('manage_subscription');

      // Plan price unit based on billing mode
      let mode = 'mensuel';
      const activeOpt = root.querySelector('.billing-option.active');
      if (activeOpt && activeOpt.dataset.billing) mode = activeOpt.dataset.billing;
      else {
        try { const saved = localStorage.getItem('billingMode'); if (saved) mode = saved; } catch {}
      }
      root.querySelectorAll('.plan-price small').forEach(s => { s.textContent = mode === 'mensuel' ? t('per_month') : t('per_year'); });

      // Placeholders for profile form
      const styleEl = root.querySelector('#user_style'); if (styleEl) styleEl.placeholder = t('style_placeholder');
      const toneEl = root.querySelector('#user_tone'); if (toneEl) toneEl.placeholder = t('tone_placeholder');
      const forbEl = root.querySelector('#user_forbidden'); if (forbEl) forbEl.placeholder = t('forbidden_placeholder');

      // Chips example texts
      const chips = root.querySelectorAll('.chips .chip');
      if (chips && chips.length >= 2) {
        chips[0].textContent = t('forbidden_chip_merci');
        chips[1].textContent = t('forbidden_chip_bravo');
      }

      // Length label and options
      const lengthLabel = root.querySelector('label[for="user_length"]'); if (lengthLabel) lengthLabel.textContent = t('length_label');
      const liShort = root.querySelector('ul.custom-select-options li[data-value="court"]'); if (liShort) liShort.textContent = t('length_short');
      const liMed = root.querySelector('ul.custom-select-options li[data-value="moyen"]'); if (liMed) liMed.textContent = t('length_medium');
      const liLong = root.querySelector('ul.custom-select-options li[data-value="long"]'); if (liLong) liLong.textContent = t('length_long');
      const lengthValue = (root.querySelector('#user_length') || {}).value || 'moyen';
      const selectedEl = root.querySelector('.custom-select-wrapper .custom-select-selected');
      if (selectedEl) {
        selectedEl.textContent = (lengthValue === 'court' ? t('length_short') : lengthValue === 'long' ? t('length_long') : t('length_medium'));
      }

      // Hint text
      const hintEl = root.querySelector('.hint'); if (hintEl) hintEl.textContent = t('hint_text');

      // Range labels
      const rLabels = root.querySelectorAll('.range-row .range-label');
      if (rLabels[0]) rLabels[0].textContent = t('linebreak_prob_label');
      if (rLabels[1]) rLabels[1].textContent = t('typo_prob_label');

      // Save buttons
      const saveBtn = root.querySelector('#comment-form button[type="submit"]'); if (saveBtn) saveBtn.textContent = t('save');
      const saveAdjustBtn = root.querySelector('#save-adjust-btn'); if (saveAdjustBtn) saveAdjustBtn.textContent = t('save_adjustments');

      // Quota (read from storage and render localized)
      const quotaEl = root.querySelector('#quota');
      if (quotaEl && typeof chrome !== 'undefined' && chrome.storage && chrome.storage.local) {
        chrome.storage.local.get(['quota','mode'], (data) => {
          const q = data && data.quota;
          const mode = data && data.mode;
          if (typeof q === 'string' && /illimit/i.test(q.normalize('NFKD'))) {
            quotaEl.textContent = (mode === 'free_trial') ? t('quota_unlimited_free_7d') : t('quota_unlimited');
          } else if (q !== undefined) {
            quotaEl.textContent = t('quota_remaining_today', { n: q });
          }
        });
      }
    } catch {}
  }

  function setLanguage(lang) {
    if (!translations[lang]) return;
    currentLang = lang;
    try { localStorage.setItem('lang', currentLang); } catch {}
    try {
      if (typeof chrome !== 'undefined' && chrome.storage && chrome.storage.local) {
        chrome.storage.local.set({ lang: currentLang });
      }
    } catch {}
    applyTranslations();
    document.dispatchEvent(new CustomEvent('i18n:languageChanged', { detail: { lang: currentLang } }));
  }

  function initLanguageSwitcher() {
    const dropdownBtn = document.getElementById('languageDropdown');
    const menu = document.getElementById('language-menu');
    if (!dropdownBtn || !menu) return;

    dropdownBtn.addEventListener('click', (e) => {
      e.stopPropagation();
      menu.style.display = (menu.style.display === 'none' || !menu.style.display) ? 'block' : 'none';
    });

    document.addEventListener('click', () => { menu.style.display = 'none'; });

    menu.querySelectorAll('[data-set-lang]').forEach(item => {
      item.addEventListener('click', (e) => {
        e.preventDefault();
        e.stopPropagation();
        const lang = item.getAttribute('data-set-lang');
        setLanguage(lang);
        menu.style.display = 'none';
      });
    });
  }

  // Expose t and setLanguage globally for convenience
  window.t = t;
  window.setLanguage = setLanguage;
  window.i18n = { t, setLanguage, applyTranslations };

  document.addEventListener('DOMContentLoaded', () => {
    initLanguageSwitcher();
    applyTranslations();
  });
})();
