async function getActiveLinkedInTab() {
  return new Promise((resolve) => {
    chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
      const t = tabs && tabs[0];
      if (t && /linkedin\.com/.test(t.url || '')) return resolve(t);
      // Try to find another LinkedIn tab
      chrome.tabs.query({ url: '*://*.linkedin.com/*' }, (ltabs) => {
        if (ltabs && ltabs.length) return resolve(ltabs[0]);
        resolve(null);
      });
    });
  });
}

function go(page) {
  const url = chrome.runtime.getURL('onboarding/' + page);
  location.href = url;
}

function sendMessageAsync(tabId, message) {
  return new Promise((resolve) => {
    try {
      chrome.tabs.sendMessage(tabId, message, (response) => {
        const err = chrome.runtime.lastError;
        if (err) {
          console.warn('sendMessage error:', err.message, message);
          return resolve(null);
        }
        console.log('sendMessage response:', message, response);
        resolve(response || null);
      });
    } catch (_) { resolve(null); }
  });
}

async function ensureInjected(tabId) {
  try {
    await chrome.scripting.executeScript({ target: { tabId }, files: ['content.js'] });
  } catch (_) {}
}

async function waitForContentReady(tabId, timeoutMs = 8000) {
  const t0 = Date.now();
  while (Date.now() - t0 < timeoutMs) {
    const pong = await sendMessageAsync(tabId, { action: 'ping' });
    if (pong && pong.status === 'ok') return true;
    await new Promise(r => setTimeout(r, 200));
  }
  return false;
}

function waitForTabComplete(tabId, timeoutMs = 20000) {
  return new Promise((resolve, reject) => {
    const t0 = Date.now();
    function handler(updatedTabId, info, tab) {
      if (updatedTabId === tabId && info.status === 'complete') {
        chrome.tabs.onUpdated.removeListener(handler);
        resolve(true);
      }
    }
    chrome.tabs.onUpdated.addListener(handler);
    const timer = setInterval(() => {
      if (Date.now() - t0 > timeoutMs) {
        chrome.tabs.onUpdated.removeListener(handler);
        clearInterval(timer);
        resolve(false);
      }
    }, 500);
  });
}

async function startProfileAnalysis() {
  const tab = await getActiveLinkedInTab();
  if (!tab) throw new Error('no_linkedin_tab');

  // Ensure content script is injected (for existing tabs)
  await ensureInjected(tab.id);
  await waitForContentReady(tab.id, 6000);

  // Ask content script for the best profile URL (with retries)
  let targetUrl = null;
  for (let i = 0; i < 4 && !targetUrl; i++) {
    const hrefRes = await sendMessageAsync(tab.id, { action: 'get_profile_href' });
    if (hrefRes && hrefRes.status === 'ok' && hrefRes.href) targetUrl = hrefRes.href;
    if (!targetUrl) await new Promise(r => setTimeout(r, 400));
  }

  // Fallback: navigate to feed then retry
  if (!targetUrl) {
    await new Promise(r => chrome.tabs.update(tab.id, { url: 'https://www.linkedin.com/feed/' }, () => r()));
    await waitForTabComplete(tab.id, 20000);
    await ensureInjected(tab.id);
    await waitForContentReady(tab.id, 6000);
    const hrefRes2 = await sendMessageAsync(tab.id, { action: 'get_profile_href' });
    if (hrefRes2 && hrefRes2.status === 'ok' && hrefRes2.href) targetUrl = hrefRes2.href;
  }

  if (!targetUrl) throw new Error('profile_href_failed');

  // Prefer the activity page to load more posts without extra navigation during extraction
  let activityUrl = null;
  try {
    const u = new URL(targetUrl);
    if (!/recent-activity/.test(u.pathname)) {
      if (!u.pathname.endsWith('/')) u.pathname += '/';
      u.pathname += 'recent-activity/all/';
    }
    activityUrl = u.toString();
  } catch { activityUrl = targetUrl; }

  // Navigate the tab if needed
  const finalUrl = activityUrl || targetUrl;
  if (finalUrl && tab.url !== finalUrl) {
    await new Promise(r => chrome.tabs.update(tab.id, { url: finalUrl }, () => r()));
    await waitForTabComplete(tab.id, 25000);
    await ensureInjected(tab.id);
    await waitForContentReady(tab.id, 6000);

    // Fallback si la page activité est 404 → revenir au profil de base
    const status = await sendMessageAsync(tab.id, { action: 'check_not_found' });
    if (status && status.notFound && targetUrl && finalUrl !== targetUrl) {
      await new Promise(r => chrome.tabs.update(tab.id, { url: targetUrl }, () => r()));
      await waitForTabComplete(tab.id, 25000);
      await ensureInjected(tab.id);
      await waitForContentReady(tab.id, 6000);
    }
  }

  // Run extraction on the loaded profile page
  const data = await sendMessageAsync(tab.id, { action: 'extract_profile_exhaustive',  options: { minPosts: 10, minComments: 10 } });
    if (!data || data.status !== 'ok') throw new Error((data && data.error) || 'extract_failed');
    return data.data;
  }

async function runAnalysisAndPersist(raw) {
  const persona = window.OcalisAnalysis.buildPersona(raw);
  await new Promise(r => chrome.storage.local.set({ persona }, r));
  return persona;
}

async function loadPersona() {
  return new Promise(res => chrome.storage.local.get(['persona'], v => res(v.persona || null)));
}

function setText(id, value) {
  const el = document.getElementById(id);
  if (el) el.textContent = value;
}

window.OcalisFlow = { getActiveLinkedInTab, go, startProfileAnalysis, runAnalysisAndPersist, loadPersona, setText };
