const bar = document.getElementById('bar');
const statusEl = document.getElementById('status');
const hint = document.getElementById('hint');

let progress = 6;
let progTimer = null;
function startSmoothProgress(target, speed=0.8) {
  if (progTimer) clearInterval(progTimer);
  progTimer = setInterval(() => {
    progress = Math.min(target, progress + Math.max(0.2, (target-progress)*0.1*speed));
    bar.style.width = progress.toFixed(0) + '%';
  }, 150);
}

function stopProgress(final=progress) {
  if (progTimer) clearInterval(progTimer);
  progress = final; bar.style.width = final.toFixed(0)+'%';
}

async function getToken() {
  return new Promise(res => chrome.storage.local.get(['accessToken'], d => res(d.accessToken || null)));
}

async function analyzeWithServer(raw) {
  const token = await getToken();
  const endpoint = 'https://app.cliaso.com/lnapi/analyze-profile';
  const headers = { 'Content-Type': 'application/json' };
  if (token) headers['Authorization'] = `Bearer ${token}`;
  const body = JSON.stringify({ profile: raw });
  const t0 = performance.now();
  const res = await fetch(endpoint, { method: 'POST', headers, body });
  const txt = await res.text();
  const elapsed = performance.now() - t0;
  let json;
  try { json = JSON.parse(txt); } catch { throw new Error('Réponse invalide: ' + txt.slice(0,200)); }
  if (!res.ok) throw new Error(json && (json.error||json.message) || 'analyse échouée');
  return { data: json, elapsed };
}

function personaToProfileData(p) {
  if (!p) return {};
  const style = (p.litteraire && p.litteraire.lexique ? p.litteraire.lexique.slice(0,8).join(', ') : '') +
                (p.litteraire && p.litteraire.structures ? ('; ' + p.litteraire.structures.join(', ')) : '');
  const tone = (p.recommandations_generation && p.recommandations_generation.variabilité) || 'neutre';
  const length = (p.recommandations_generation && p.recommandations_generation.longueur) || '1 à 3 phrases';
  const forbidden = (p.interdits || []).join(', ');
  return { style, tone, length, forbidden };
}

(async () => {
  try {
    statusEl.textContent = (typeof t === 'function' ? t('ob_status_opening_profile') : 'Ouverture du profil...');
    startSmoothProgress(25, 1.0);
    const tStart = performance.now();
    const raw = await OcalisFlow.startProfileAnalysis();
    const tMid = performance.now();
    const extractMs = tMid - tStart;

    statusEl.textContent = (typeof t === 'function' ? t('ob_status_cleaning_texts') : 'Nettoyage des textes...');
    startSmoothProgress(45, 0.7);

    statusEl.textContent = (typeof t === 'function' ? t('ob_status_ai_analysis') : 'Analyse intelligente avec une IA...');
    startSmoothProgress(90, 0.5);
    const { data, elapsed } = await analyzeWithServer(raw);

    await new Promise(r => chrome.storage.local.set({ persona: data }, r));
    try { localStorage.setItem('profileData', JSON.stringify(personaToProfileData(data.persona))); } catch {}

    statusEl.textContent = (typeof t === 'function' ? t('ob_status_finalizing') : 'Finalisation...');
    startSmoothProgress(100, 1.2);
    setTimeout(() => { stopProgress(100); OcalisFlow.go('step3_result.html'); }, 500);

  } catch (e) {
    stopProgress(0);
    statusEl.textContent = (typeof t === 'function' ? t('ob_error_cannot_analyze') : 'Impossible d’analyser.');
    hint.classList.remove('hidden');
    console.error(e);
  }
})();

