﻿const API_BASE = 'https://app.cliaso.com/lnapi'; 
let profileData = {};

// Storage helpers (promise)
function getStorage(keys) {
  return new Promise(res => chrome.storage.local.get(keys, data => res(data)));
}
function setStorage(obj) {
  return new Promise(res => chrome.storage.local.set(obj, () => res()));
}
function removeStorage(keys) {
  return new Promise(res => chrome.storage.local.remove(keys, () => res()));
}

// UI helper
function $(id){ return document.getElementById(id); }

// i18n helpers for dynamic messages without touching legacy strings
function normalizeText(s){
  try {
    return String(s||'')
      .normalize('NFKD')
      .replace(/[\u0300-\u036f]/g,'')       // strip diacritics
      .replace(/[^\w\s]/g,' ')              // strip punctuation/emoji
      .replace(/\s+/g,' ')                   // collapse spaces
      .toLowerCase().trim();
  } catch(_) { return String(s||'').toLowerCase().trim(); }
}
const FR_TO_KEY = {
  'inscription reussie.': 'register_success',
  "impossible de s'inscrire (erreur reseau)": 'register_network_error',
  'connexion reussie.': 'login_success',
  'mots de passe ou email incorrects.': 'login_invalid_credentials',
  'reponse invalide du serveur.': 'invalid_server_response',
  "impossible de se connecter (erreur reseau).": 'login_network_error',
  'tous les champs sont requis.': 'all_fields_required',
  'email invalide.': 'invalid_email',
  'mot de passe faible. minimum 8 caracteres, avec majuscule, minuscule, chiffre et caractere special.': 'weak_password_message',
  'les mots de passe ne correspondent pas.': 'passwords_mismatch',
  'numero de telephone invalide.': 'invalid_phone',
  'veuillez accepter les conditions generales.': 'accept_terms_required',
  "un compte avec cet email existe deja.": 'register_user_exists',
  'veuillez remplir tous les champs.': 'register_missing_fields',
  "erreur lors de l'inscription. veuillez reessayer.": 'register_generic_error',
  'email et mot de passe requis.': 'email_and_password_required',
  'veuillez vous reconnecter.': 'please_relogin',
  'etes-vous sur de lancer ocalis?': 'confirm_launch',
  'veuillez valider le profil ia avant de lancer les reponses.': 'validate_profile_reply_required',
  'veuillez valider le profil ia avant de lancer les commentaires.': 'validate_profile_comment_required',
  'veuillez activer au moins une action.': 'enable_at_least_one_action',
  'ajustements sauvegardes.': 'adjustments_saved',
  'reglage ia enregistre.': 'ai_settings_saved',
  "aucune url de paiement recue.": 'no_checkout_url_received',
  'erreur lors de la mise a niveau.': 'upgrade_error',
  'portail indisponible.': 'portal_unavailable',
  "erreur lors de l'acces au portail.": 'portal_access_error',
  "veuillez vous connecter.": 'please_login',
  "votre quota journalier est maintenant epuise (0 commentaire a envoyer). vous pourrez revenir demain.": 'quota_depleted_alert'
};
// Build normalized lookup to be resilient to punctuation/emoji/diacritics
const FR_TO_KEY_NORM = {};
try { Object.keys(FR_TO_KEY).forEach(k => { FR_TO_KEY_NORM[normalizeText(k)] = FR_TO_KEY[k]; }); } catch(_){ }
function translateMessage(msg){
  try {
    if (typeof window.t === 'function'){
      const cleaned = String(msg || '').replace(/^[^\w]+/u, '');
      const key = FR_TO_KEY_NORM[normalizeText(cleaned)];
      if (key) return t(key);
    }
  } catch(_){}
  return msg;
}

// Monkey-patch alert/confirm to translate known FR messages
try {
  const __origAlert = window.alert.bind(window);
  window.alert = (m) => __origAlert(translateMessage(m));
  const __origConfirm = window.confirm.bind(window);
  window.confirm = (m) => __origConfirm(translateMessage(m));
} catch(_){}

// ---------- AUTH FUNCTIONS ----------
async function apiFetch(path, opts = {}) {
  // auto-attach auth header if accessToken present
  const st = await getStorage(['accessToken', 'refreshToken']);
  let headers = opts.headers || {};
  if (st.accessToken) headers['Authorization'] = `Bearer ${st.accessToken}`;
  headers['Content-Type'] = headers['Content-Type'] || 'application/json';
  const res = await fetch(API_BASE + path, { ...opts, headers });

  if (res.status === 401) {
    // try refresh once
    if (st.refreshToken) {
      const r = await fetch(API_BASE + '/auth/refresh', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ refreshToken: st.refreshToken })
      });
      if (r.ok) {
        const j = await r.json();
        await setStorage({ accessToken: j.accessToken });
        headers['Authorization'] = `Bearer ${j.accessToken}`;
        return fetch(API_BASE + path, { ...opts, headers });
      } else {
        await removeStorage(['accessToken','refreshToken','user']);
        renderAuthState(); // update UI
        return res;
      }
    }
  }
  return res;
}

async function register({ name, email, password, phone }) {
  try {
    const r = await fetch(API_BASE + '/auth/register', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ name, email, password, phone })
    });

    const json = await r.json();

    if (r.ok) {
      showNotification("Inscription réussie.", "success");
      const url = chrome.runtime.getURL('onboarding/step1_welcome.html');
      chrome.tabs.create({ url });
      window.close();
    } 

    return json;
  } catch (err) {
    showNotification("Impossible de s'inscrire (erreur réseau)", "error"); 
    throw err;
  }
}

function logToBackground(data, prefix) {
  chrome.runtime.sendMessage({ type: "log", data, prefix });
}

async function login(email, password) {
  const url = API_BASE + '/auth/login';

  try {
    const r = await fetch(url, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ email, password })
    });

    const text = await r.text();

    try {
      const json = JSON.parse(text);

      if (r.ok) {
        showNotification("Connexion réussie.", "success");
        const url = chrome.runtime.getURL('onboarding/step1_welcome.html');
        chrome.tabs.create({ url });
        window.close();
      } else {
        showNotification("Mots de passe ou email incorrects.", "error");
      }

      return json;
    } catch (e) {
      showNotification("Réponse invalide du serveur.", "error");
      throw new Error("Invalid JSON response, got: " + text.slice(0, 100));
    }
  } catch (err) {
    showNotification("Impossible de se connecter (erreur réseau).", "error");
    throw err;
  }
}

async function logout() {
  const st = await getStorage(['refreshToken']);
  if (st.refreshToken) {
    await fetch(API_BASE + '/auth/logout', {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({ refreshToken: st.refreshToken })
    });
    
  }

  await removeStorage(['accessToken','refreshToken','user']);
  renderAuthState();
}

async function fetchMe() {
  try {
    const r = await apiFetch('/auth/me');
    if (!r.ok) return null;
    const j = await r.json();
    return j.user;
  } catch (err) { return null; }
}

// ---------- UI / Auth rendering ----------
async function renderAuthState() {
  const st = await getStorage(['accessToken','refreshToken','user']);
  const authForms = $('auth-forms');
  const authInfo  = $('auth-info');
  const mainUi    = $('main-ui');  

  if (st.accessToken && st.user) {
    if (authForms) authForms.style.display = 'none';
    if (authInfo) authInfo.style.display = 'block';
    if (mainUi) mainUi.style.display = 'grid';

    $('user-email').textContent = st.user.email;
    $('user-email').title = st.user.email;
  } else {
    if (authForms) authForms.style.display = 'block';
    if (authInfo) authInfo.style.display = 'none';
    if (mainUi) mainUi.style.display = 'none';

    const loginView = document.getElementById('auth-login');
    const registerView = document.getElementById('auth-register');
    if (loginView && registerView) {
      loginView.style.display = 'block';
      registerView.style.display = 'none';
    }
  }
}


// ---------- Hook UI events ----------
function wireAuthUI() {

  const loginView = document.getElementById('auth-login');
  const registerView = document.getElementById('auth-register');

  const switchToLogin = () => {
    if (!loginView || !registerView) return;
    loginView.style.display = 'block';
    registerView.style.display = 'none';
  };
  const switchToRegister = () => {
    if (!loginView || !registerView) return;
    loginView.style.display = 'none';
    registerView.style.display = 'block';
  };
  switchToLogin();

  document.getElementById('auth-show-register').addEventListener('click', (e) => {
    e.preventDefault();
    switchToRegister();
  });

  document.getElementById('auth-show-login').addEventListener('click', (e) => {
    e.preventDefault();
    switchToLogin();
  });

  // ---------- Validators ----------
  const isValidEmail = (email) => /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(email);
  const isValidPassword = (pw) => /^(?=.*[A-Z])(?=.*[a-z])(?=.*\d)(?=.*[\W_]).{8,}$/.test(pw);
  const isValidPhone = (phone) => /^\+?[0-9\s\-]{7,15}$/.test(phone);

  // ---------- Register handler ----------
  document.getElementById('auth-register-submit').addEventListener('click', async () => {
    const name = document.getElementById('reg-identifier').value.trim();
    const pw = document.getElementById('reg-password').value;
    const pwConfirm = document.getElementById('reg-password-confirm').value;
    const email = document.getElementById('reg-email').value.trim();
    const phone = document.getElementById('reg-phone').value.trim();
    const terms = document.getElementById('reg-terms');

    // Vérifications
    if (!name || !email || !pw || !pwConfirm || !phone) {
      return showNotification("Tous les champs sont requis.", "error");
    }
    if (!isValidEmail(email)) {
      return showNotification("Email invalide.", "error");
    }
    if (!isValidPassword(pw)) {
      return showNotification("Mot de passe faible. Minimum 8 caractères, avec majuscule, minuscule, chiffre et caractère spécial.", "error");
    }
    if (pw !== pwConfirm) {
      return showNotification("Les mots de passe ne correspondent pas.", "error");
    }
    if (!isValidPhone(phone)) {
      return showNotification("Numéro de téléphone invalide.", "error");
    }
    if (terms && !terms.checked) {
      return showNotification("Veuillez accepter les conditions générales.", "error");
    }

    // Appel API
    const j = await register({ name, email, password: pw, phone });

    if (j.error) {
      if (j.error === 'user_exists') {
        return showNotification("Un compte avec cet email existe déjà.", "error");
      } else if (j.error === 'missing_fields') {
        return showNotification("Veuillez remplir tous les champs.", "error");
      } else {
        return showNotification("Erreur lors de l'inscription. Veuillez réessayer.", "error");
      }
    }

    await setStorage({ accessToken: j.accessToken, refreshToken: j.refreshToken, user: j.user });
    renderAuthState();
  });

  // ---------- Login handler ----------
  document.getElementById('auth-login-submit').addEventListener('click', async () => {
    const email = document.getElementById('login-email').value.trim();
    const pw = document.getElementById('login-password').value;

    if (!email || !pw) return showNotification("Email et mot de passe requis.", "error");
    if (!isValidEmail(email)) return showNotification("Email invalide.", "error");

    const j = await login(email, pw);
    if (j.error) return console.error("Erreur connexion: ", j.error);

    await setStorage({ accessToken: j.accessToken, refreshToken: j.refreshToken, user: j.user });
    renderAuthState();
  });

  document.getElementById('logout-btn').addEventListener('click', async () => {
    await logout();
  });
}

async function verifyLinkedInTab() {
  chrome.tabs.query({active: true, currentWindow: true}, (tabs) => {
    const body = document.body;
    if (!tabs || !tabs[0] || !tabs[0].url.startsWith("https://www.linkedin.com")) {
      body.innerHTML = `
        <div style="
          display: flex;
          flex-direction: column;
          align-items: center;
          justify-content: center;
          height: 100vh;
          text-align: center;
          font-family: 'Segoe UI', Roboto, Arial, sans-serif;
        ">
          <div style="
            background: #fff3cd;
            color: #856404;
            border: 1px solid #ffeeba;
            padding: 20px 25px;
            border-radius: 12px;
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
            max-width: 300px;
          ">
            <h4 style="margin-bottom: 12px;">âš ï¸ Page LinkedIn introuvable</h4>
            <p>Ouvrir LinkedIn pour utiliser OCALIS.</p>
            <button id="openBtn" style="
              margin-top: 12px;
              padding: 8px 16px;
              border: none;
              border-radius: 6px;
              background-color: #231971;
              color: #fff;
              font-weight: 500;
              cursor: pointer;
              transition: background 0.2s;
            ">Ouvrir ici</button>
          </div>
        </div>
      `;

      document.getElementById('openBtn').addEventListener('click', () => {
        chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
          if (tabs && tabs[0]) {
            chrome.tabs.update(tabs[0].id, { url: "https://www.linkedin.com/feed" });
            window.close(); 
          }
        });
      });      

      return false; 
    }
    return true;
  });
}

async function init() {
  //await verifyLinkedInTab();
  wireAuthUI();
  // check token validity
  const st = await getStorage(['accessToken','refreshToken','user']);
  if (st.accessToken) {
    const me = await fetchMe();
    if (me) {
      // store user info
      await setStorage({ user: me });
    } else {
      // try refresh
      if (st.refreshToken) {
        const r = await fetch(API_BASE + '/auth/refresh', {
          method: 'POST', headers: { 'Content-Type': 'application/json' }, body: JSON.stringify({ refreshToken: st.refreshToken })
        });
        if (r.ok) {
          const j = await r.json();
          await setStorage({ accessToken: j.accessToken });
          const me2 = await fetchMe();
          if (me2) await setStorage({ user: me2 });
        } else {
          // tokens invalid
          await removeStorage(['accessToken','refreshToken','user']);
        }
      } else {
        await removeStorage(['accessToken','refreshToken','user']);
      }
    }
  }
  renderAuthState();
  wireMainUI(); 
  wireSubscriptionUI();
  loadSubscriptionStatus();
}

function runContentFunction(action, count=1, profileData={}) {
  chrome.tabs.query({active: true, currentWindow: true}, (tabs) => {
    if (!tabs || !tabs[0]) return;
    chrome.storage.local.get(['accessToken','user'], async (st) => {
      if (!st.accessToken) { showNotification("Veuillez vous connecter.", "error"); return; }
      chrome.tabs.sendMessage(
        tabs[0].id,
        { action, count, profileData, auth: { accessToken: st.accessToken } },
        (response) => { console.log("Réponse de content.js :", response); }
      );
    });
  });
}

function wireMainUI() {
  const launchBtn = $('launch-btn');
const stopBtn = $('stop-btn');
const likeToggle = $('auto-like-toggle');
const commentToggle = $('auto-comment-toggle');
const replyToggle = $('auto-reply-toggle');
const replyMyPostsOnly = document.getElementById('reply-my-posts-only');
const profileSettings = $('profile-settings');
const saveAdjustBtn = $('save-adjust-btn');
const commentForm = $('comment-form');
const refreshBtn = $('refresh-plan-btn');

// Sliders and live labels
const exRange = $('exclamation_prob');
const qRange  = $('question_prob');
const dRange  = $('dot_prob');
const typoRange = $('typo_prob');

const crLabel = document.getElementById('crlf_prob_label');
const tLabel  = document.getElementById('typo_prob_label');

document.querySelectorAll('.custom-select-wrapper').forEach(wrapper => {
  const selected = wrapper.querySelector('.custom-select-selected');
  const options = wrapper.querySelector('.custom-select-options');
  const input = wrapper.querySelector('input[type="hidden"]');

  selected.addEventListener('click', () => {
    options.style.display = options.style.display === 'block' ? 'none' : 'block';
  });

  options.querySelectorAll('li').forEach(li => {
    li.addEventListener('click', () => {
      input.value = li.dataset.value;
      selected.textContent = li.textContent;
      options.querySelectorAll('li').forEach(l => l.classList.remove('selected'));
      li.classList.add('selected');
      options.style.display = 'none';
      // trigger event si besoin
      input.dispatchEvent(new Event('change'));
    });
  });

  // close if click outside
  document.addEventListener('click', (e) => {
    if (!wrapper.contains(e.target)) options.style.display = 'none';
  });
});

const toPct = (v) => `${Math.round((parseFloat(v || 0)) * 100)}%`;

function updateRangeLabels() {
  if (exRange && crLabel) crLabel.textContent = toPct(exRange.value);
  if (typoRange && tLabel) tLabel.textContent = toPct(typoRange.value);
}

function logSliders() {
  console.log("Sliders:", {
    exclamation_prob: exRange ? exRange.value : null,
    question_prob: qRange ? qRange.value : null,
    dot_prob: dRange ? dRange.value : null,
    typo_prob: typoRange ? typoRange.value : null
  });
}

// --- Synchronisation des 3 sliders CRLF (UI seulement) ---
function syncCRLFSliders(source, value) {
  if (exRange && source !== exRange) exRange.value = value;
  if (qRange && source !== qRange) qRange.value = value;
  if (dRange && source !== dRange) dRange.value = value;
  updateRangeLabels();
  //logSliders();
}

// écouteurs sliders
if (exRange) exRange.addEventListener('input', (e) => syncCRLFSliders(exRange, e.target.value));
if (qRange)  qRange.addEventListener('input', (e) => syncCRLFSliders(qRange, e.target.value));
if (dRange)  dRange.addEventListener('input', (e) => syncCRLFSliders(dRange, e.target.value));

if (typoRange) typoRange.addEventListener('input', () => { updateRangeLabels(); });

// initialize once at load
updateRangeLabels();
//logSliders();

function prefillProfileForm({ forceOpen = false } = {}) {
  const saved = localStorage.getItem('profileData');
  if (!saved) return false;

  try {
    const data = JSON.parse(saved);
    $('user_style').value = data.style || '';
    $('user_tone').value = data.tone || '';
    $('user_length').value = data.length || 'moyen';
    $('user_forbidden').value = data.forbidden || '';

    $('exclamation_prob').value = data.exclamationProb || 0;
    $('question_prob').value    = data.questionProb || 0;
    $('dot_prob').value         = data.dotProb || 0;

    if (typoRange) typoRange.value = data.typoProb || 0;

    if (forceOpen) {
      commentToggle.checked = true;
      profileSettings.style.display = 'block';
    }
    profileData = data;

    // refresh slider labels after we set values
    try { updateRangeLabels(); } catch {}
    return true;
  } catch {
    return false;
  }
}

prefillProfileForm();

function saveTogglesState() {
  const state = {
    like: likeToggle.checked,
    comment: commentToggle.checked,
    reply: replyToggle ? replyToggle.checked : false,
    replyMyPostsOnly: replyMyPostsOnly ? replyMyPostsOnly.checked : false
  };
  localStorage.setItem('togglesState', JSON.stringify(state));
}

function loadTogglesState() {
  const saved = localStorage.getItem('togglesState');
  if (!saved) return;
  try {
    const state = JSON.parse(saved);
    likeToggle.checked = !!state.like;
    commentToggle.checked = !!state.comment;
    if (replyToggle) replyToggle.checked = !!state.reply;
    if (replyMyPostsOnly) replyMyPostsOnly.checked = !!state.replyMyPostsOnly;

    if (commentToggle.checked) {
      profileSettings.style.display = 'block';
    } else {
      profileSettings.style.display = 'none';
    }
    updateReplyOptionsVisibility();
  } catch {}
}
loadTogglesState();

likeToggle.addEventListener('change', saveTogglesState);
commentToggle.addEventListener('change', (e) => {
  // Mutual exclusivity: if comment ON → reply OFF
  if (e.target.checked && replyToggle) {
    replyToggle.checked = false;
  }
  saveTogglesState();
  updateReplyOptionsVisibility();
  updateProfileSettingsVisibility();
});
if (replyToggle) replyToggle.addEventListener('change', (e) => {
  // Mutual exclusivity: if reply ON → comment OFF
  if (e.target.checked) {
    commentToggle.checked = false;
  }
  saveTogglesState();
  updateReplyOptionsVisibility();
  updateProfileSettingsVisibility();
});
if (replyMyPostsOnly) replyMyPostsOnly.addEventListener('change', saveTogglesState);

function saveProfileData(updates) {
  try {
    const saved = localStorage.getItem('profileData');
    if (saved) profileData = JSON.parse(saved);
  } catch {}

  // fusionne updates
  profileData = { ...profileData, ...updates };

  localStorage.setItem('profileData', JSON.stringify(profileData));
  console.log("profileData après fusion :", localStorage.getItem('profileData'));
}


saveAdjustBtn.addEventListener('click', () => {
  const updates = {
    exclamationProb: exRange ? exRange.value : 0,
    questionProb: qRange ? qRange.value : 0,
    dotProb: dRange ? dRange.value : 0,
    typoProb: typoRange ? typoRange.value : 0
  };

  saveProfileData(updates);

    chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
    if (!tabs || !tabs[0]) return;
    chrome.tabs.sendMessage(
      tabs[0].id,
      { action: 'saveProfile', profileData: JSON.parse(localStorage.getItem('profileData')) },
      () => {
        showNotification("Ajustements sauvegardés.", "success");
        profileSettings.style.display = 'none';
      }
    );
  });
});


commentForm.addEventListener('submit', (e) => {
  e.preventDefault();

  const updates = {
    style: $('user_style').value,
    tone: $('user_tone').value,
    length: $('user_length').value,
    forbidden: $('user_forbidden').value
  };

  saveProfileData(updates);

  chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
    if (!tabs || !tabs[0]) return;
    chrome.tabs.sendMessage(
      tabs[0].id,
      { action: 'saveProfile', profileData: JSON.parse(localStorage.getItem('profileData')) },
      () => {
        showNotification("Réglage IA enregistré.", "success");
        profileSettings.style.display = 'none';
      }
    );
  });
});


launchBtn.addEventListener('click', async () => {
  const st = await getStorage(['accessToken', 'user']);
  if (!st.accessToken) { 
    showNotification("Veuillez vous reconnecter.", "error"); 
    return; 
  }

  if (!confirm("Etes-vous sûr de lancer Ocalis?")) {
    return; 
  }
  
  const count = $('count').value;

  const like = likeToggle.checked, comment = commentToggle.checked;
  const reply = replyToggle ? replyToggle.checked : false;
  if (reply) {
    if (!profileData) { 
      showNotification("Veuillez valider le profil IA avant de lancer les réponses.", "error"); 
      return; 
    }
    const options = { myPostsOnly: replyMyPostsOnly ? replyMyPostsOnly.checked : false };
    runReplySniperWithNavigation(count, profileData, options);
  }
  else if (like && comment) runContentFunction('likeComment', count, profileData);
  else if (like) runContentFunction('like', count);
  else if (comment) {
    if (!profileData) { 
      showNotification("Veuillez valider le profil IA avant de lancer les commentaires.", "error"); 
      return; 
    }
    runContentFunction('comment', count, profileData);
  } else {
    showNotification("Veuillez activer au moins une action.", "error");
  }
});    

stopBtn.addEventListener('click', () => {
  chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
    if (!tabs || !tabs[0]) return;
    chrome.tabs.get(tabs[0].id, (tab) => {
      chrome.storage.local.get(['accessToken'], (st) => {
        chrome.tabs.sendMessage(
          tabs[0].id,
          { action: 'stop', auth: { accessToken: st.accessToken } },
          (response) => { console.log('Réponse stop content.js :', response); }
        );
      });
    });
  });
});

refreshBtn.addEventListener("click", async () => {
  refreshBtn.disabled = true;
  refreshBtn.textContent = "⏳...";
  try {
    await loadSubscriptionStatus();
  } finally {
    refreshBtn.disabled = false;
    refreshBtn.innerHTML = `<i class="fa-solid fa-rotate-right"></i>`;
  }
});
}

init();

function showNotification(message, type = "info", duration = 3000) {
  try { message = translateMessage(message); } catch(_){ }
  const container = document.getElementById("notification-container");
  if (!container) return;

  const notif = document.createElement("div");
  notif.textContent = message;

  notif.style.cssText = `
    padding: 10px 16px;
    border-radius: 8px;
    font-size: 14px;
    font-family: 'Segoe UI', Roboto, Arial, sans-serif;
    box-shadow: 0 4px 8px rgba(0,0,0,0.1);
    animation: fadeIn 0.3s ease forwards;
    cursor: pointer;
  `;

  if (type === "success") {
    notif.style.background = "#d4edda";
    notif.style.color = "#155724";
    notif.style.border = "1px solid #c3e6cb";
  } else if (type === "error") {
    notif.style.background = "#f8d7da";
    notif.style.color = "#721c24";
    notif.style.border = "1px solid #f5c6cb";
  } else {
    notif.style.background = "#cce5ff";
    notif.style.color = "#004085";
    notif.style.border = "1px solid #b8daff";
  }

  container.appendChild(notif);

  notif.addEventListener("click", () => notif.remove());

  setTimeout(() => {
    notif.style.animation = "fadeOut 0.5s ease forwards";

    notif.addEventListener("animationend", () => {
      notif.remove();
    }, { once: true }); 
  }, duration);
}

// Localize quota text if present
function localizeQuotaText(){
  try {
    const el = document.getElementById('quota');
    if (!el) return;
    const txt = (el.textContent || '').toString();
    const nMatch = txt.match(/(\d+)/);
    if (typeof t === 'function'){
      if (/illimit/i.test(txt)) {
        // keep as is; handled elsewhere if needed
        return;
      }
      if (nMatch) {
        el.textContent = t('quota_remaining_today', { n: nMatch[1] });
      }
    }
  } catch {}
}
try { document.addEventListener('i18n:languageChanged', localizeQuotaText); } catch {}

async function loadSubscriptionStatus() {
  const card = $('subscription-card');
  card.style.display = 'block';

  const st = await getStorage(['accessToken', 'refreshToken']);

  if (!st.accessToken) return;
  
  try {
    const res = await apiFetch('/subscription-status');
    const data = await res.json();

    $('current-plan').textContent = data.plan || 'gratuit';

    const billingRaw = data.billing || data.billingCycle || data.interval;
    let billingNormalized = billingRaw;
    if (billingRaw === 'monthly') billingNormalized = 'mensuel';
    if (billingRaw === 'yearly' || billingRaw === 'annual') billingNormalized = 'annuel';
    if (billingNormalized === 'mensuel' || billingNormalized === 'annuel') {
      try { localStorage.setItem('billingMode', billingNormalized); } catch (_) {}
    }

    if (data.currentPeriodEnd) {
      const date = new Date(data.currentPeriodEnd).toLocaleDateString("fr-FR");
      $('plan-expiry').textContent = `(Expire le : ${date})`;
    } else {
      $('plan-expiry').textContent = '';
    }

    if (data.plan === 'gratuit' || data.plan === 'none') {
      $('upgrade-btn').style.display = 'block';
      $('manage-sub-btn').style.display = 'none';
    } else {
      $('upgrade-btn').style.display = 'block';
      $('manage-sub-btn').style.display = 'block';
    }

  } catch (err) {
    console.error("Erreur loadSubscriptionStatus:", err);
  }
}


function wireSubscriptionUI() {
  const upgradeBtn = document.getElementById('upgrade-btn');
  const modal = document.getElementById('plan-modal');
  const closeModal = document.getElementById('close-plan-modal');
  const billingOptions = modal.querySelectorAll('.billing-option');
  const planCards = modal.querySelectorAll('.plan-card');
  let billingMode = 'mensuel';

  const setBillingMode = (mode) => {
    billingMode = mode;
    billingOptions.forEach(opt => {
      opt.classList.toggle('active', opt.dataset.billing === mode);
    });

    planCards.forEach(card => {
      const priceEl = card.querySelector('.plan-price span');
      const smallEl = card.querySelector('.plan-price small');
      let monthlyInfo = card.querySelector('.plan-monthly-info');

      if (!monthlyInfo) {
        monthlyInfo = document.createElement('div');
        monthlyInfo.className = 'plan-monthly-info';
        monthlyInfo.style.fontSize = '15px';
        monthlyInfo.style.color = '#555';
        monthlyInfo.style.marginTop = '2px';
        card.querySelector('.plan-price').appendChild(monthlyInfo);
      }

      const plan = card.dataset.plan;
      let price = 0;
      let monthly = 0;

      if (plan === 'pro') { price = (mode === 'mensuel' ? 29 : 174); monthly = 15; }
      if (plan === 'elite') { price = (mode === 'mensuel' ? 89 : 534); monthly = 45; }
      if (plan === 'gold') { price = (mode === 'mensuel' ? 149 : 894); monthly = 75; }

      priceEl.textContent = price + "\u00A0€";
      smallEl.textContent = mode === 'mensuel' ? '/mois' : '/an';
      monthlyInfo.textContent = mode === 'annuel' ? `(≈ ${monthly.toFixed(0)} € / mois)` : '';
    });

    try { localStorage.setItem('billingMode', billingMode); } catch (_) {}
    try { if (window.i18n && typeof window.i18n.applyTranslations === 'function') window.i18n.applyTranslations(); } catch (_) {}
  };

  const loadBillingMode = () => {
    try {
      const saved = localStorage.getItem('billingMode');
      if (saved === 'mensuel' || saved === 'annuel') billingMode = saved;
    } catch (_) {}
    setBillingMode(billingMode);
  };

  loadBillingMode();

  billingOptions.forEach(opt => {
    opt.addEventListener('click', () => setBillingMode(opt.dataset.billing));
  });

  upgradeBtn.addEventListener('click', () => {
    loadBillingMode();
    modal.style.display = 'block';
  });

  closeModal.addEventListener('click', () => {
    modal.style.display = 'none';
  });

  planCards.forEach(btn => {
    btn.addEventListener('click', async () => {
      const chosenPlan = btn.dataset.plan;
      const payload = { plan: chosenPlan, billing: billingMode };

      try {
        const st = await getStorage(['accessToken','refreshToken','user']);
        payload.user = st.user;

        const res = await apiFetch('/create-checkout-session', {
          method: 'POST',
          body: JSON.stringify(payload)
        });

        const data = await res.json();

        if (data.url) {
          window.open(data.url, '_blank');
        } else if (data.message) {
          showNotification(data.message, "success");
          loadSubscriptionStatus();
        } else {
          showNotification("Aucune URL de paiement reçue.", "warning");
        }

      } catch (err) {
        console.error("Erreur upgrade:", err);
        showNotification("Erreur lors de la mise à niveau.", "error");
      } finally {
        modal.style.display = 'none';
      }
    });
  });

  document.getElementById('manage-sub-btn').addEventListener('click', async () => {
    try {
      const res = await apiFetch('/create-portal-session', { method: 'POST' });
      const data = await res.json();
      if (data.url) window.open(data.url, '_blank');
      else showNotification("Portail indisponible.", "warning");
    } catch (err) {
      console.error("Erreur portail:", err);
      showNotification("Erreur lors de l'accès au portail.", "error");
    }
  });
}

let lastQuota = null;
let alertTimeout = null;

const observer = new MutationObserver(() => {
  const quotaEl = document.getElementById("quota");
  if (quotaEl && !quotaEl.dataset.bound) {
    quotaEl.dataset.bound = "true";

    // Quota initial
    chrome.storage.local.get(["quota", "mode"], (data) => {
      if (data.quota !== undefined) {
        if (typeof data.quota === "string" && data.quota.startsWith("illimité")) {
          quotaEl.textContent = 
            data.mode === "free_trial"
              ? "Quota : illimité gratuitement pendant 7 jours🚀"
              : "Quota : illimité 🥇";
          lastQuota = "illimité";
        } else {
          quotaEl.textContent = `Quota restant aujourd'hui : ${data.quota} commentaire(s)`;
          lastQuota = data.quota;
        }
      }
    });

    chrome.storage.onChanged.addListener((changes, area) => {
      if (area === "local" && (changes.quota || changes.mode)) {
        const newQuota = changes.quota?.newValue;
        const newMode = changes.mode?.newValue;

        if (typeof newQuota === "string" && newQuota.startsWith("illimité")) {
          quotaEl.textContent =
            newMode === "free_trial"
              ? "Quota : illimité pendant l'essai gratuit 🚀"
              : "Quota : illimité 🥇";
          lastQuota = "illimité";
          if (alertTimeout) clearTimeout(alertTimeout);
          return;
        }

        quotaEl.textContent = `Quota restant aujourd'hui : ${newQuota} commentaire(s)`;

        if (typeof lastQuota === "number" && lastQuota > 0 && newQuota === 0) {
          if (alertTimeout) clearTimeout(alertTimeout);

          alertTimeout = setTimeout(() => {
            alert("⚠️ Votre quota journalier est maintenant épuisé (0 commentaire à envoyer). Vous pourrez revenir demain.");
          }, 10000);
        }

        lastQuota = newQuota;
      }
    });
  }
});

observer.observe(document.body, { childList: true, subtree: true });


// Localized quota rendering (overlays existing binding with i18n strings)
function __renderQuotaLocalized(quota, mode){
  try {
    const el = document.getElementById('quota'); if (!el) return;
    const isUnlimited = (typeof quota === 'string') && /illimit/i.test(quota.normalize('NFKD'));
    if (isUnlimited){
      el.textContent = (typeof t === 'function')
        ? (mode === 'free_trial' ? t('quota_unlimited_free_7d') : t('quota_unlimited'))
        : (mode === 'free_trial' ? 'Quota : illimité gratuitement pendant 7 jours' : 'Quota : illimité');
      return;
    }
    const n = (typeof quota === 'number') ? quota : Number(quota||0);
    el.textContent = (typeof t === 'function') ? t('quota_remaining_today', { n }) : `Quota restant aujourd'hui : ${n} commentaire(s)`;
  } catch {}
}

(function __bindQuotaI18n(){
  try {
    if (window.__ocalisQuotaI18nBound) return; window.__ocalisQuotaI18nBound = true;
    // Initial render
    chrome.storage.local.get(['quota','mode'], (data)=> __renderQuotaLocalized(data.quota, data.mode));
    // Update on changes
    chrome.storage.onChanged.addListener((changes, area) => {
      if (area !== 'local') return;
      const q = (changes.quota && changes.quota.newValue);
      const m = (changes.mode && changes.mode.newValue);
      if (q !== undefined || m !== undefined){
        chrome.storage.local.get(['quota','mode'], (data)=> __renderQuotaLocalized(data.quota, data.mode));
      }
    });
    // Update on language switch
    document.addEventListener('i18n:languageChanged', () => {
      chrome.storage.local.get(['quota','mode'], (data)=> __renderQuotaLocalized(data.quota, data.mode));
    });
  } catch {}
})();


// ====== Reply Sniper launcher with new-tab navigation ======
function waitForTabComplete(tabId, timeoutMs = 25000) {
  return new Promise((resolve) => {
    const t0 = Date.now();
    function handler(updatedTabId, info) {
      if (updatedTabId === tabId && info.status === 'complete') {
        chrome.tabs.onUpdated.removeListener(handler);
        resolve(true);
      }
    }
    chrome.tabs.onUpdated.addListener(handler);
    const timer = setInterval(() => {
      if (Date.now() - t0 > timeoutMs) {
        chrome.tabs.onUpdated.removeListener(handler);
        clearInterval(timer);
        resolve(false);
      }
    }, 500);
  });
}

function sendMessageAsync(tabId, message){
  return new Promise((resolve)=>{
    try{
      chrome.tabs.sendMessage(tabId, message, (response)=>{
        const err = chrome.runtime.lastError; if (err) return resolve(null);
        resolve(response||null);
      });
    }catch(_){ resolve(null); }
  });
}

async function ensureInjected(tabId){
  try { await chrome.scripting.executeScript({ target: { tabId }, files: ['content.js'] }); } catch(_){}
}

async function waitForContentReady(tabId, timeoutMs = 8000){
  const t0 = Date.now();
  while (Date.now() - t0 < timeoutMs){
    const pong = await sendMessageAsync(tabId, { action:'ping' });
    if (pong && pong.status === 'ok') return true;
    await new Promise(r=>setTimeout(r,200));
  }
  return false;
}

async function sendWithRetry(tabId, msg, attempts = 12){
  for (let i=0;i<attempts;i++){
    const resp = await sendMessageAsync(tabId, msg);
    if (resp) return resp;
    console.warn('[replySniper-launcher] retry', i+1, 'because response is null');
    await ensureInjected(tabId);
    await waitForContentReady(tabId, 6000);
    await new Promise(r=>setTimeout(r, 500));
  }
  return null;
}

async function runReplySniperWithNavigation(count, profileData, options){
  chrome.storage.local.get(['accessToken','user'], async (st) => {
    if (!st.accessToken) { showNotification("Veuillez vous connecter.", "error"); return; }

    const profileUrl = 'https://www.linkedin.com/in/';

    chrome.tabs.query({active: true, currentWindow: true}, async (tabs) => {
      const active = tabs && tabs[0];
      const wantProfile = !!options.myPostsOnly;

      const isOnProfile = active && /\/in\//.test(active.url||'');

      // Decide whether to reuse current tab or open a new one
      if ((wantProfile && isOnProfile)) {
        // Run in current tab
        await ensureInjected(active.id);
        await waitForContentReady(active.id, 8000);
        const msg = { action:'replySniper', count, profileData, options, auth:{ accessToken: st.accessToken } };
        const resp = await sendWithRetry(active.id, msg, 12);
        console.log('replySniper (same tab) response:', resp);
        return;
      }

      chrome.tabs.create({ url: profileUrl, active: true }, async (newTab) => {
        // Wait for load complete
        await waitForTabComplete(newTab.id, 30000);
        await ensureInjected(newTab.id);
        await waitForContentReady(newTab.id, 8000);
        const msg = { action:'replySniper', count, profileData, options, auth:{ accessToken: st.accessToken } };
        const resp = await sendWithRetry(newTab.id, msg, 12);
        console.log('replySniper (new tab) response:', resp);
      });
    });
  });
}

function updateReplyOptionsVisibility(){
  const row = document.getElementById('reply-options');
  if (!row) return;
  const rt = document.getElementById('auto-reply-toggle');
  row.style.display = (rt && rt.checked) ? 'flex' : 'none';
}

function updateProfileSettingsVisibility(){
  const el = document.getElementById('profile-settings');
  if (!el) return;
  const ct = document.getElementById('auto-comment-toggle');
  const rt = document.getElementById('auto-reply-toggle');
  const anyOn = (ct && ct.checked) || (rt && rt.checked);
  el.style.display = anyOn ? 'block' : 'none';
}
/*
// ------ Onboarding quick access (opens new tab with guided flow) ------
function insertOnboardingShortcut() {
  try {
    if (document.getElementById('onboarding-shortcut')) return;
    const cta = document.createElement('button');
    cta.id = 'onboarding-shortcut';
    cta.textContent = 'Créer mon double IA';
    cta.style.cssText = 'position:fixed; right:14px; top:10px; z-index:9999; background:#f39b3d;color:#fff;border:none;border-radius:10px;padding:6px 10px;font-weight:700;box-shadow:0 6px 14px rgba(243,155,61,.35);cursor:pointer;';
    cta.addEventListener('click', () => {
      const url = chrome.runtime.getURL('onboarding/step1_welcome.html');
      chrome.tabs.create({ url });
      window.close();
    });
    document.body.appendChild(cta);
  } catch {}
}

document.addEventListener('DOMContentLoaded', insertOnboardingShortcut);
*/



